﻿#pragma once

#include  "Invoke.hpp"

class EventHandle;

/// プロセッサから進行状況に関する通知を受けるインターフェース。
/// 一般に、進行状況ダイアログクラスが本クラスを継承してインターフェースを実装する。
class ProcessUI
{
public:

  /// プロセッサのスレッドハンドルを知らせるメソッド。
  /// Parameters:
  ///   hThread - プロセッサのスレッドハンドル。
  virtual void NotifyThread(HANDLE hThread) = 0;

  /// 中止および終了を知らせるイベントを設定するメソッド。
  /// Parameters:
  ///   pQuitEvent - ProcessUI 側でユーザーが中止の指示をした場合に、プロセッサ側に通知するためのイベント。
  ///   pDoneEvent - 正常終了／強制終了を問わず、プロセッサ側が処理を終えたことを通知するためのイベント。ProcessUI 側は、このイベントがセットされた UI を廃棄して処理を終了してよい。
  virtual void SetEvents(EventHandle *pQuitEvent, EventHandle *pDoneEvent) = 0;

  /// 進行状況の最大値を設定するメソッド。
  virtual void SetMaximum(u64 progressMax) = 0;

  /// 進行状況の現在値を設定するメソッド。
  virtual void SetCurrent(u64 progressCur) = 0;

  /// 進行状況の現在値を指定した値だけ増加させるメソッド。
  virtual void AddCurrent(u64 delta) = 0;

  /// 進行中の処理の名称を設定するメソッド。
  /// Parameters:
  ///   process - 進行中の処理名を表す短い文字列。空文字列が渡されることはない。
  virtual void SetProcess(const szstring &process) = 0;

  /// 進行中の処理のコンテナとアイテムの名前を設定するメソッド。
  /// Parameters:
  ///   container - 進行中のコンテナの名称等を表す文字列。
  ///   item - 進行中のアイテムの名称等を表す文字列。
  virtual void SetItem(const szstring &container, const szstring &item) = 0;

  /// UI スレッドで指定された関数を実行するメソッド。.NET の Invoke と似た動作をする。
  virtual LRESULT Invoke(InvokedFunction func, void * param) = 0;

  virtual ~ProcessUI() = 0;

};
